#!/bin/bash

# Get the current directory
script_dir=$(dirname "$(readlink -f "$0")")
app_folder="${script_dir}/app"
tmp_folder="${script_dir}/tmp"

# Step 1: Extract version from mapbot-0.6.jar
local_version=$(find "${script_dir}" -type f -name 'mapbot-*.jar' | grep -oP 'mapbot-\K[0-9]+\.[0-9]+' | head -n1)

# Step 2: Get the text from currentVersion.txt
remote_version=$(curl -s https://www.mapbot.gg/currentVersion.txt)

# Function to remove whitespaces from a string
remove_whitespace() {
  echo "$1" | tr -d '[:space:]'
}

# Remove whitespaces and compare
cleaned_local_version=$(remove_whitespace "$local_version")
cleaned_remote_version=$(remove_whitespace "$remote_version")

# Display the versions
echo "Local version: ${cleaned_local_version}"
echo "Remote version: ${cleaned_remote_version}"

# Step 3: Compare versions
if [ "$cleaned_local_version" != "$cleaned_remote_version" ]; then
    zip_file="Mapbot-${remote_version}.zip"

    # Step 4: Download if the zip file does not exist
    if [ ! -f "${zip_file}" ]; then
        echo "Local version is older. Downloading..."
        curl -sSLO https://www.mapbot.gg/${zip_file}
        echo "Download completed!"
    else
        echo "Zip file already exists locally. Skipping download."
    fi

    # Step 5: Unzip to tmp folder
    echo "Unzipping to temporary folder..."
    unzip -q "${zip_file}" -d "${tmp_folder}"

    # Step 6: Delete old JAR file
    old_jar="${app_folder}/mapbot-${local_version}.jar"
    if [ -f "${old_jar}" ]; then
        echo "Deleting old JAR file: ${old_jar}"
        rm "${old_jar}"
    fi

    # Step 7: Copy tmp/Mapbot/app folder contents to the main app folder
    echo "Copying contents to the app folder..."
    if [ -d "${tmp_folder}/Mapbot/app" ]; then
        cp -rf "${tmp_folder}/Mapbot/app/." "${app_folder}"
        echo "Update completed!"
    else
        echo "Error: Unable to find 'Mapbot/app' folder in the extracted contents."
    fi

    # Step 8: Delete old Mapbot.exe if it exists
    mapbot_exe="${script_dir}/Mapbot.exe"
    if [ -f "${mapbot_exe}" ]; then
        echo "Deleting old Mapbot.exe: ${mapbot_exe}"
        rm -f "${mapbot_exe}"  # Use -f option to force deletion without confirmation
    fi

    # Step 9: Replace Mapbot.exe with /tmp/Mapbot/Mapbot.exe
    tmp_mapbot_exe="${tmp_folder}/Mapbot/Mapbot.exe"
    if [ -f "${tmp_mapbot_exe}" ]; then
        echo "Copying new Mapbot.exe from /tmp/Mapbot/..."
        cp "${tmp_mapbot_exe}" "${script_dir}"
        echo "Copy completed!"
    else
        echo "Error: /tmp/Mapbot/Mapbot.exe not found."
    fi

    # Step 10: Copy tmp/Mapbot/app folder contents to the main app folder
    echo "Copying contents to the tessdata folder..."
    if [ -d "${tmp_folder}/Mapbot/tessdata" ]; then
        cp -rf "${tmp_folder}/Mapbot/tessdata/." "${app_folder}"
        echo "Update completed!"
    else
        echo "Error: Unable to find 'Mapbot/tessdata' folder in the extracted contents."
    fi

    # Clean up temporary files
    rm -rf "${tmp_folder}"
else
    echo "Local version is up-to-date. No update needed."
fi

# Prompt user to press 'Enter' to close
echo "Press 'Enter' to close..."
read
